<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserWalletHistory;
use App\Rules\CheckGuardOperation;
use App\Services\MyService;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): object
    {
        try {

            $users = User::all()
                ->map(function ($user) {

                    $date = explode(' ', MyService::G2J($user->created_at));

                    return [
                        'id' => $user->id,
                        'username' => $user->username,
                        'mobile' => $user->mobile,
                        'is_blocked' => $user->is_blocked,
                        'created' => $date[0],
                        'state' => $user->state?->name,
                        'city' => $user->city?->name,
                        'wallet' => $user->wallet,
                    ];
                });


            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $users
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): object
    {

        try {
            $validator = Validator::make($request->all(), [
                'mobile' => ['required', 'regex:/^09\d{9}$/', 'unique:users,mobile'],
                'username' => ['required', 'string', 'max:255'],
                'state_id' => ['required', 'integer', Rule::exists('states', 'id')->whereNull('parent_id'),],
                'city_id' => ['required', Rule::exists('states', 'id')->where('parent_id', $request->state_id),],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }


            $user = User::create([
                'mobile' => $request->mobile,
                'username' => $request->username,
                'state_id' => $request->state_id,
                'city_id' => $request->city_id
            ]);

            $token = $user->createToken('auth_token', ['user'])->plainTextToken;

            return response()->json([
                'success' => true,
                'message' => 'ثبت ‌نام با موفقیت انجام شد',
                'token' => $token,
                'uid' => $user->id
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): object
    {
        try {

            $validator = Validator::make(['id' => $id], [
                'id' => ['required', 'integer', 'exists:users,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            try {
                MyService::checkRealPerson($id, 'user');
            } catch (\Exception $e) {
                return $e->response();
            }

            $user = User::find($id);

            unset($user['fcm_token'], $user['remember_token']);

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $user
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'username' => ['required', 'string', 'min:3', 'max:255'],
                'is_blocked' => ['sometimes', 'boolean', new CheckGuardOperation],
                'wallet' => ['sometimes', 'numeric', new CheckGuardOperation],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            try {
                MyService::checkRealPerson($id, 'user');
            } catch (\Exception $e) {
                return $e->response();
            }

            DB::beginTransaction();

            $data = $validator->validated();

            $createHistory = true;

            if (isset($data['wallet'])) {
               $createHistory = MyService::newUserWalletRecord($id, $data['wallet'], 'تغییر از پنل مدیریت');
            }

           $updateUser = User::find($id)->update($data);

            if (!$updateUser || !$createHistory) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'عملیات انجام نشد !'
                ]);
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
