<?php

namespace App\Http\Controllers\User;

use App\Enums\TravelStage;
use App\Events\NewRequestDriver;
use App\Http\Controllers\Controller;
use App\Models\AppConfig;
use App\Models\DriverCategories;
use App\Models\DriverUser;
use App\Models\TravelDestinations;
use App\Models\TravelDetails;
use App\Models\Travels;
use App\Rules\ValidDestinationServiceArea;
use App\Rules\ValidDiscountCode;
use App\Rules\ValidOriginServiceArea;
use App\Services\MyService;
use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class TravelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): object
    {
        $user = auth('user')->user();
        $uid = $user->id;

        $runningTravels = Travels::where('uid', $uid)
            ->where('is_running', 1)
            ->with(['driver', 'destinations'])
            ->get()
            ->map(function ($travel) use ($user) {
                $runningDestination = $travel->destinations->where('is_running', 1)->first() ?? $travel->destinations->first();

                $estimatedTime = MyService::estimatedDrivingTime(
                    $travel->driver?->current_lat,
                    $travel->driver?->current_lng,
                    $runningDestination?->lat,
                    $runningDestination?->lng,
                );

                $res = (object)[
                    'driver' => (object)[
                        'name' => $travel->driver?->first_name . ' ' . $travel->driver?->last_name,
                        'mobile' => $travel->driver?->mobile,
                        'profile' => $travel->driver?->profile,
                        'vehicle_plate' => $travel->driver?->vehicle_plate,
                        'vehicle' => $travel->driver?->vehicle_name . ' ' . $travel->driver?->vehicle_color,
                    ],
                    'details_route' => (object)[
                        'route' => MyService::travelCurrentRoute($travel),
                        'estimated_time' => $estimatedTime['estimated_time']
                    ],
                    'payment' => (object)[
                        'user_wallet' => $user?->wallet,
                        'travel_price' => $travel->travel_price,
                        'app_commission' => $travel->app_commission,
                        'total_price' => round($travel->travel_price + $travel->app_commission),
                    ],
                    'info' => (object)[
                        'cancel_button' => !$travel->has_started,
                        'travel_id' => $travel->id,
                        'travel_code' => $travel->travel_code,
                    ],
                    'details_travel' => $travel->details ?? (object)[],
                ];

                $res->details_travel->delay_time = $travel->delay_time;
                $res->details_travel->commuting = (bool)$travel->commuting;
                $res->details_travel->two_destination = ($travel->commuting && count($travel->destinations) > 2)
                    || (!$travel->commuting && count($travel->destinations) > 1);

                return $res;
            });


        $historyTravels = Travels::where('uid', $uid)
            ->where(function ($q) {
                $q->where('is_completed', 1)->orWhere('is_canceled', 1);
            })
            ->get()
            ->map(function ($travel) {

                $weekDay = MyService::weekDays($travel->created_at);
                $date = explode(' ', MyService::G2J($travel->created_at));

                $payment = $travel->payment;

                if ($payment) {
                    if ($payment === 'cash') {
                        $payment = 'نقدی';
                    }

                    if ($payment === 'online') {
                        $payment = 'آنلاین';
                    }
                }

                return [
                    'date' => $date[0],
                    'date_time' => $date[1],
                    'weekDay' => $weekDay,
                    'status' => $travel->is_completed ? 'موفق' : 'لغو شده',
                    'color_status' => $travel->is_completed ? '#C0FFF3' : '#FFC6C6',
                    'driver_category' => (object)[
                        'name' => $travel->driverCategory->name ?? 'نامشخص',
                        'pic' => $travel->driverCategory->pic,
                    ],
                    'total_price' => $travel->travel_price + $travel->app_commission,
                    'cancel_type' => $travel->cancel_type,
                    'cancel_description' => $travel->cancel_description,
                    'payment_type' => $payment,
                    'route' => $travel->destinations->map(function ($destination) {

                        $path = $destination->order === 1 ? 'از' : 'به';

                        return [
                            'position' => $path,
                            'address' => $destination->destination_address,
                        ];
                    })
                ];
            });


        return response()->json([
            'success' => true,
            'message' => 'عملیات با موفقیت انجام شد',
            'response' => [
                'running' => $runningTravels,
                'history' => $historyTravels,
            ],
        ]);
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'registration' => ['required', 'boolean'],
                'discount_code' => ['nullable', 'string', new ValidDiscountCode()],
                'origin_lat' => ['required', 'numeric', 'between:-90,90'],
                'origin_lng' => ['required', 'numeric', 'between:-180,180'],
                'origin_address' => ['required', 'string', 'max:255'],
                'driver_category' => ['required', 'integer', 'exists:driver_categories,id'],
                'destinations' => ['required', 'array', 'min:1', 'max:2'],
                'destinations.*.lat' => ['required', 'numeric', 'between:-90,90'],
                'destinations.*.lng' => ['required', 'numeric', 'between:-180,180'],
                'destinations.*.address' => ['required', 'string', 'max:255'],
                'delay_time' => ['nullable', 'numeric', 'between:0,60'],
                'commuting' => ['nullable', 'boolean'],

//                'recipient_name' => ['sometimes', 'required_if:driver_category,2,3', 'string', 'max:255'],
//                'mobile' => ['sometimes', 'required_if:driver_category,2,3', 'numeric', 'digits:11'],
//                'address' => ['sometimes', 'required_if:driver_category,2,3', 'string', 'max:255'],
//                'value_price' => ['sometimes', 'required_if:driver_category,2,3', 'string', 'max:255'],
//                'load_type_id' => ['sometimes', 'required_if:driver_category,3', 'integer', 'exists:load_types,id'],
//                'payment_recipient' => ['sometimes', 'required_if:driver_category,2,3', 'boolean'],
//                'description' => ['nullable', 'string', 'max:255'],
            ]);


            $validator->after(function ($validator) use ($request) {

                if ($validator->errors()->has('origin_lat') || $validator->errors()->has('origin_lng')) {
                    return;
                }

                $lat = (float)$request->input('origin_lat');
                $lng = (float)$request->input('origin_lng');
                $originRule = null;

                if ($lat && $lng) {
                    $originRule = new ValidOriginServiceArea($lat, $lng);

                    $originRule->validate('origin_location', null, function ($message) use ($validator) {
                        $validator->errors()->add('origin_lat', $message);
                    });

                }


                $allowInterProvinceTravel = Cache::rememberForever('traveling_outside_province', function () {
                    return AppConfig::where('title', 'traveling_outside_province')->value('content') ?? 20;
                });

                $destination = $request->input('destinations');

                if ($destination && $originRule) {
                    foreach ($request->input('destinations') as $index => $dest) {
                        $destinationRule = new ValidDestinationServiceArea(
                            $dest['lat'],
                            $dest['lng'],
                            $originRule->getCity(),
                            $allowInterProvinceTravel
                        );

                        $destinationRule->validate("destinations.$index.lat", null, function ($message) use ($validator, $index) {
                            $validator->errors()->add("destinations.$index.lat", $message);
                        });
                    }
                }
            });

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            DB::beginTransaction();

            $originLat = (float)$request->input('origin_lat');
            $originLng = (float)$request->input('origin_lng');
            $destinations = $request->input('destinations');
            $commuting = (bool)$request->input('commuting', false);

            $coordinates = [
                [$originLng, $originLat],
            ];

            foreach ($destinations as $item) {
                $coordinates[] = [(float)$item['lng'], (float)$item['lat']];
            }

            if ($commuting) {
                $coordinates[] = [$originLng, $originLat];
            }

            try {
                $detailsTrip = MyService::calculateTripDistanceViaORS($coordinates, $commuting);
            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 403);
            }

            $driverCategory = DriverCategories::findOrFail($request->input('driver_category'));

            if (!$driverCategory->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => "درحال حاضر دسته بندی $driverCategory->name در دسترس نیست !"
                ], 403);
            }

            $totalPrice = MyService::calculateTripTotalPrice($driverCategory, $detailsTrip['total']['distance_km']);

            if ($request->input('discount_code')) {
                try {
                    $calculateDiscount = MyService::calculateDiscountCode($request->input('discount_code'), $totalPrice, $request->input('registration'));
                    $totalPrice -= $calculateDiscount;
                } catch (\Exception $e) {
                    DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => $e->getMessage()
                    ], 403);
                }
            }

            if ($request->input('delay_time') && $request->input('delay_time') > 0) {
                $delayPrice = MyService::calculateDelayTime((int)$request->input('delay_time'));
                $totalPrice += $delayPrice;
            }


            $commission = Cache::rememberForever('app_commission', function () {
                return AppConfig::where('title', 'app_commission')->value('content') ?? 20;
            });


            $app_commission = $totalPrice * ($commission / 100);

            $detailsTrip['total']['price'] = round($totalPrice + $app_commission);

            // registration 👇🏻

            if ($request->input('registration')) {

                $user = auth('user')->user();

                if ($user?->is_blocked) {
                    return response()->json([
                        'success' => false,
                        'message' => 'پروفایل شما درحال حاضر مسدود شده است و مجاز به درخواست سفر نمی باشید !'
                    ], 403);
                }

                $checkTravel = Travels::where([
                    'uid' => auth('user')->id(),
                    'is_canceled' => 0,
                    'is_completed' => 0,
                    'is_unsuccessful' => 0,
                ])->latest()->first();

                if ($checkTravel) {
                    return response()->json([
                        'success' => false,
                        'message' => 'در حال حاضر شما یک سفر تمام نشده دارید و قادر به درخواست مجدد سفر نمی باشید !',
                        'travel_id' => $checkTravel->id
                    ], 403);
                }

                $validator->sometimes([
                    'recipient_name',
                    'address',
                    'value_price',
                ], ['required', 'string', 'max:255'], function ($input) {
                    return in_array($input->driver_category, [2, 3], true);
                });

                $validator->sometimes('mobile', ['numeric', 'digits:11'], function ($input) {
                    return in_array($input->driver_category, [2, 3], true);
                });

                $validator->sometimes('payment_recipient', ['required', 'boolean'], function ($input) {
                    return in_array($input->driver_category, [2, 3], true);
                });

                $validator->sometimes('load_type_id', ['required', 'integer', 'exists:load_types,id'], function ($input) {
                    return $input->driver_category == 3;
                });

                if ($validator->fails()) {
                    return response()->json([
                        'success' => false,
                        'message' => $validator->errors()->first()
                    ], 422);
                }

                $travelData = $request->only(['discount_code', 'origin_lat', 'origin_lng', 'origin_address', 'driver_category', 'delay_time', 'commuting']);
                $travelData['uid'] = auth('user')->id();
                $travelData['travel_code'] = MyService::generateUniqueTravelCode();
                $travelData['travel_price'] = $totalPrice;
                $travelData['app_commission'] = $app_commission;
                $travelData['payment_method'] = 'cash';

                $createTravel = Travels::create($travelData);

                if (!$createTravel) {
                    DB::rollBack();
                    throw new \Exception('متاسفانه مشکلی در ایجاد سفر پیش آمده است  !');
                }

                $detailsTrip['travel_id'] = $createTravel->id;

                if ($commuting) {
                    $destinations[] = [
                        'lat' => $originLat,
                        'lng' => $originLng,
                        'address' => $request->input('origin_address'),
                    ];
                }


                foreach ($destinations as $index => $dest) {
                    $dest['order'] = $index + 1;
                    $dest['travel_id'] = $createTravel->id;
                    $dest['distance_km'] = $detailsTrip['routes'][$index]['distance_km'];
                    $dest['destination_address'] = $dest['address'];

                    $createdDestination = TravelDestinations::create($dest);

                    if (!$createdDestination) {
                        DB::rollBack();
                        throw new \Exception('متاسفانه مشکلی در پردازش مقصد پیش آمده است  !');
                    }

                    $detailsTrip['routes'][$index]['dest_id'] = $createdDestination->id;
                }


                if ($request->input('driver_category') === 2 || $request->input('driver_category') === 3) {
                    $dataDetailsTravel = $request->only(['recipient_name', 'mobile', 'address', 'description', 'value_price', 'payment_recipient']);

                    if ($request->input('driver_category') === 3) $dataDetailsTravel['load_type_id'] = $request->input('load_type_id');
                    $dataDetailsTravel['travel_id'] = $createTravel->id;

                    $createdDetails = TravelDetails::create($dataDetailsTravel);

                    if (!$createdDetails) {
                        DB::rollBack();
                        throw new \Exception('متاسفانه مشکلی در پردازش جزئیات پیش آمده است  !');
                    }
                }


                $drivers = MyService::getDriversNearby($travelData['driver_category'], $originLat, $originLng);

                foreach ($drivers as $driver) {
                        $detailsTravel = $createTravel->details ?? (object)[];

                        $detailsTravel->delay_time = $createTravel->delay_time;
                        $detailsTravel->commuting = (bool)$createTravel->commuting;
                        $detailsTravel->two_destination = ($createTravel->commuting && count($createTravel->destinations) > 2) || (!$createTravel->commuting && count($createTravel->destinations) > 1);

//                        broadcast(new NewRequestDriver([
//                            'driver_id' => $driver->id,
//                            'travel_id' => $createTravel->id,
//                            'travel_code' => $createTravel->travel_code,
//                            'origin_time' => $driver->currentDriver['estimated_time'],
//                            'routes' => $detailsTrip['routes'],
//                            'total' => $detailsTrip['total'],
//                            'details_travel' => $detailsTravel,
//                            'user' => (object)[
//                                'username' => $user->username,
//                                'mobile' => $user->mobile,
//                            ],
//                            'status' => MyService::travelStatus($createTravel)
//                        ]));

                }


                DB::commit();
                return response()->json([
                    'success' => true,
                    'registration' => true,
                    'message' => 'عملیات با موفقیت انجام شد',
                    'response' => $detailsTrip
                ]);
            }


            return response()->json([
                'success' => true,
                'registration' => false,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $detailsTrip
            ]);


        } catch (Exception $e) {
            DB::rollBack();
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
