<?php

namespace App\Http\Controllers\User;

use App\Events\CancelRequestFortDriver;
use App\Events\RemoveRequest;
use App\Http\Controllers\Controller;

use App\Models\CommentsAboutDriver;
use App\Models\Travels;
use App\Services\MyService;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class MiscController extends Controller
{
    public function travelCanceled(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'cancel_title' => ['required', 'string', 'max:255'],
                'cancel_description' => ['nullable', 'string', 'max:255'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $uid = auth('user')->id();

            $travel = Travels::where([
                'uid' => $uid,
                'is_canceled' => 0,
                'is_completed' => 0,
                'is_unsuccessful' => 0,
            ])->latest()->first();


            if (!$travel) {
                return response()->json([
                    'success' => false,
                    'message' => 'درحال حاضر سفر فعالی ندارید !'
                ], 403);
            }


            if ($travel->has_started) {
                return response()->json([
                    'success' => false,
                    'message' => 'سفر شما درحال انجام است و قابلیت لغو سفر امکان پذیر نمی باشد !'
                ], 403);
            }


            $cancelTitle = $request->input('cancel_title');
            $cancelDescription = $request->input('cancel_description');

            if ($cancelDescription) {
                $cancelTitle .= ' ' . '-' . ' ' . $cancelDescription;
            }

            $travel->cancel_description = $cancelTitle;
            $travel->is_canceled = 1;
            $travel->is_running = 0;
            $travel->cancel_type = 'user';
            $travel->save();

            if ($travel->driver_id) {

                $travel->driver->is_available = 1;
                $travel->driver->save();

                MyService::socketCancelRequestFortDriver($travel, 'کاربر سفر خود را به علت ' . $cancelTitle . ' لغو کرد !');

            } else {

                $drivers = MyService::getDriversNearby($travel->driver_category, $travel->origin_lat, $travel->origin_lng);

                foreach ($drivers as $driver) {
                    MyService::socketRemoveRequest($travel, $driver->id);
                }

            }

            return response()->json([
                'success' => true,
                'message' => 'با موفقیت لغو شد' . ' ' . $travel->travel_code . ' ' . 'سفر شما با کد'
            ]);


        } catch (Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    public function travelRunning()
    {
        try {

//            Example is rating 👇🏻

//            return response()->json([
//                'success' => true,
//                'message' => 'شما یک سفر در حال انجام دارید',
//                'is_running' => false,
//                'is_searching' => false,
//                'is_rating' => true,
//                'response' => [
//                    'driver_id' => 1,
//                    'travel_id' => 29,
//                    'driver_name' => 'هادی انصاری'
//                ],
//            ]);

            $user = auth('user')->user();

            $travel = Travels::where([
                'uid' => $user?->id,
                'is_canceled' => 0,
                'is_completed' => 0,
                'is_unsuccessful' => 0,
            ])->latest()->first();

            if (!$travel) {
                return response()->json([
                    'success' => true,
                    'message' => 'در حال حاضر سفری ندارید',
                    'is_running' => false,
                    'is_searching' => false,
                    'is_rating' => false,
                ]);
            }


            if ($travel->is_running) {

                $runningDestination = $travel->destinations->where('is_running', 1)->first() ?? $travel->destinations[0];

                $estimatedTime = MyService::estimatedDrivingTime(
                    $travel->driver?->current_lat,
                    $travel->driver?->current_lng,
                    $runningDestination?->lat,
                    $runningDestination?->lng,
                );

                $res = (object)[
                    'driver' => (object)[
                        'name' => $travel->driver?->first_name . ' ' . $travel->driver?->last_name,
                        'mobile' => $travel->driver?->mobile,
                        'profile' => $travel->driver?->profile,
                        'vehicle_plate' => $travel->driver?->vehicle_plate,
                        'vehicle' => $travel->driver?->vehicle_name . ' ' . $travel->driver?->vehicle_color,
                        'current_lat' => $travel->driver?->current_lat,
                        'current_lng' => $travel->driver?->current_lng,
                    ],

                    'details_route' => (object)[
                        'route' => MyService::travelCurrentRoute($travel),
                        'estimated_time' => $estimatedTime['estimated_time']
                    ],

                    'payment' => (object)[
                        'user_wallet' => $user?->wallet,
                        'travel_price' => $travel->travel_price,
                        'app_commission' => $travel->app_commission,
                        'total_price' => round($travel->travel_price + $travel->app_commission),
                        'method' => $travel->payment_method,
                    ],

                    'info' => (object)[
                        'cancel_button' => !$travel->has_started,
                        'travel_id' => $travel->id,
                        'travel_code' => $travel->travel_code,
                    ],

                    'details_travel' => $travel->details ?? (object)[],

                    'positions' => (object)[
                        'driver' => (object)[
                            'lat' => $travel->driver?->current_lat,
                            'lng' => $travel->driver?->current_lng,
                        ],

                        'origin' => (object)[
                            'lat' => $travel->origin_lat,
                            'lng' => $travel->origin_lng,
                        ],

                        'destinations' => $travel->destinations->map(function ($destination) {
                            return [
                                'lat' => $destination->lat,
                                'lng' => $destination->lng,
                            ];
                        })

                    ]
                ];


                $res->details_travel->delay_time = $travel->delay_time;
                $res->details_travel->commuting = (bool)$travel->commuting;
                $res->details_travel->two_destination = ($travel->commuting && count($travel->destinations) > 2) || (!$travel->commuting && count($travel->destinations) > 1);


                return response()->json([
                    'success' => true,
                    'message' => 'شما یک سفر در حال انجام دارید',
                    'is_running' => true,
                    'is_searching' => false,
                    'is_rating' => false,
                    'response' => $res,
                ]);
            }


//         جای اشتباه در تابع باعث میشه هیچ موقع به این نتیجه نرسه و اینکه باید چک بشه قبلا برای این سفر نظری وجود نداشته باشه

//            $travelRating = Travels::where([
//                'uid' => $user?->id,
//                'is_completed' => true,
//            ])->latest()->first();

//            if ($travelRating) {
//                $today = Carbon::today();
//                $travelDate = $travelRating->created_at;
//                $diff = $today->diffInDays($travelDate);
//
//                if ($diff === 0) {
//                    return response()->json([
//                        'success' => true,
//                        'message' => 'لطفا نظر خود را راجب سفرتان بیان کنید',
//                        'is_running' => false,
//                        'is_searching' => false,
//                        'is_rating' => true,
//                        'response' => [
//                            'driver_id' => $travelRating->driver_id,
//                            'travel_id' => $travelRating->id,
//                            'driver_name' => $travelRating->driver?->first_name . ' ' . $travelRating->driver?->last_name,
//                        ],
//                    ]);
//                }
//            }
//
            return response()->json([
                'success' => true,
                'message' => 'شما یک سفر در حال جستوجوی راننده دارید',
                'is_running' => false,
                'is_searching' => true,
                'is_rating' => false,
                'response' => (object)[
                    'travel_id' => $travel->id,
                    'travel_code' => $travel->travel_code,
                ],
            ]);

        } catch (Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    public function commentAboutDriver(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'did' => ['required', 'integer', 'exists:driver_users,id'],
                'travel_id' => ['required', 'integer', 'exists:travels,id'],
                'comment' => ['nullable', 'string', 'max:1000'],
                'stars' => ['nullable', 'integer', 'min:0', 'max:5'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $travel = Travels::find($request['travel_id']);

            if (!$travel->is_completed) {
                return response()->json([
                    'success' => false,
                    'message' => 'درحال حاضر قادر به ثبت نظر نمی باشید !'
                ], 403);
            }

            $data = $validator->validated();
            $data['uid'] = auth('user')->id();

            CommentsAboutDriver::create($data);

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد'
            ]);


        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

}
