<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Chats;
use App\Models\Travels;
use App\Services\MyService;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ChatsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {

            $validator = Validator::make($request->all(), [
                'travel_id' => ['required', 'integer', 'exists:travels,id'],
                'message' => ['required', 'string', 'min:1', 'max:1000'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }


            $data = $validator->validated();
            $travel = Travels::find($data['travel_id']);

            if ($travel->is_canceled || $travel->is_complated || !$travel->driver_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'این سفر قابلیت چت کردن ندارد !'
                ], 403);
            }


            $guard = null;
            $id = null;

            if (auth('user')->check()) {
                $guard = 'user';
                $id = auth('user')->id();

            } elseif (auth('driver')->check()) {
                $guard = 'driver';
                $id = auth('driver')->id();

            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }

            if ($guard === 'user' && $travel->uid !== $id) {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }

            if ($guard === 'driver' && $travel->driver_id !== $id) {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }

            $data['uid'] = $travel->uid;
            $data['did'] = $travel->driver_id;
            $data['user_type'] = $guard;

            if (!Chats::create($data)) {
                return response()->json([
                    'success' => false,
                    'message' => 'متاسفانه مشکلی پیش آمده است !'
                ]);
            }

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد'
            ]);


        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {

            $validator = Validator::make(['id' => $id], [
                'id' => ['required', 'integer', 'exists:travels,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $travel = Travels::find($id);


            if (auth('user')->check()) {
                $guard = 'user';
                $id = auth('user')->id();

            } elseif (auth('driver')->check()) {
                $guard = 'driver';
                $id = auth('driver')->id();

            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }

            if ($guard === 'user' && $travel->uid !== $id) {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }

            if ($guard === 'driver' && $travel->driver_id !== $id) {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }


            $chats = Chats::where(['travel_id' => $id, 'uid' => $travel->uid, 'did' => $travel->driver_id])
                ->get()
                ->map(function ($chat) {

                    $date = explode(' ', MyService::G2J($chat->created_at));
                    $chat->time = $date[1];

                    return $chat;
                });

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $chats,
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
