<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\UserAddresses;
use App\Rules\ValidOriginServiceArea;
use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AddressesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index() :object
    {
        try {

            $uid = auth('user')->id();
            $addresses = UserAddresses::where('uid', $uid)->get();

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $addresses
            ]);

        }catch (Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request) :object
    {
        try {

            $validator = Validator::make($request->all(), [
                'title' => ['required', 'string', 'max:255'],
                'address' => ['required', 'string', 'max:255'],
                'lat' => ['required', 'numeric', 'between:-90,90'],
                'lng' => ['required', 'numeric', 'between:-180,180'],
            ]);

            $validator->after(function ($validator) use ($request) {
                $lat = (float)$request->input('lat');
                $lng = (float)$request->input('lng');

                if ($lat && $lng) {
                    $originRule = new ValidOriginServiceArea($lat, $lng);

                    $originRule->validate('location', null, function ($message) use ($validator) {
                        $validator->errors()->add('location', $message);
                    });

                }
            });

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $address  = $validator->validated();
            $address['uid'] = auth('user')->id();

            UserAddresses::create($address);

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
            ]);

        }catch (Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id) :object
    {
        try {

            $validator = Validator::make(['id' => $id], [
                'id' => ['required', 'integer', 'exists:user_addresses,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $address = UserAddresses::find($id);
            $uid = auth('user')->id();

            if ($address->uid !== $uid) {
                return response()->json([
                    'success' => false,
                    'message' => 'آدرس انتخاب شده متعلق به شخص دیگری می باشد !'
                ] , 403);
            }

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $address
            ]);

        }catch (Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id) :object
    {
        try {

            $validator = Validator::make(['id' => $id], [
                'id' => ['required', 'integer', 'exists:user_addresses,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $address = UserAddresses::find($id);
            $uid = auth('user')->id();

            if ($address->uid !== $uid) {
                return response()->json([
                    'success' => false,
                    'message' => 'آدرس انتخاب شده متعلق به شخص دیگری می باشد !'
                ] , 403);
            }

            $address->delete();

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
            ]);

        }catch (Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }
}
