<?php

namespace App\Http\Controllers\Payment;

use App\Enums\PaymentActions;
use App\Enums\PaymentsStatus;
use App\Http\Controllers\Controller;
use App\Models\Payments;
use App\Models\Travels;
use App\Services\MyService;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class PaymentController extends Controller
{
    /**
     * @throws ValidationException
     */
    public function newPayment(Request $request): object
    {

        try {

            $validator = Validator::make($request->all(), [
                'device' => ['required', 'string', 'in:web,android'],
                'action' => ['required', 'string', 'in:user_wallet,driver_wallet,travel_payment'],
                'travel_id' => ['required_if:action,travel_payment', 'integer', 'exists:travels,id'],
                'amount' => ['required_if:action,user_wallet,driver_wallet', 'numeric', 'min:1000'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $action = $validator->validated()['action'];

            if (($action === PaymentActions::UserWallet ->value|| $action === PaymentActions::TravelPayment->value) && !auth('user')->check()) {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }

            if ($action === PaymentActions::DriverWallet->value && !auth('driver')->check()) {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }

            $data['action'] = $action;

            switch ($action) {
                case PaymentActions::UserWallet->value:
                    $data = [
                        'action' => PaymentActions::UserWallet,
                        'amount' => $validator->validated()['amount'],
                        'device' => $validator->validated()['device'],
                        'uid' => auth('user')->id(),
                    ];
                    break;

                case PaymentActions::TravelPayment->value:
                    $travel = Travels::findOrFail($validator->validated()['travel_id']);

                    $data = [
                        'action' => PaymentActions::TravelPayment,
                        'travel_id' => $travel->id,
                        'amount' => $travel->travel_price + $travel->app_commission,
                        'device' => $validator->validated()['device'],
                        'uid' => auth('user')->id(),
                        'driver_id' => $travel->driver_id,
                    ];
                    break;

                case PaymentActions::DriverWallet->value:
                    $data = [
                        'action' => PaymentActions::DriverWallet,
                        'amount' => $validator->validated()['amount'],
                        'device' => $validator->validated()['device'],
                        'driver_id' => auth('driver')->id(),
                    ];
                    break;

                default:
                    $data = [];
            }

            $createdPayment = Payments::create($data);

            return MyService::zarinpalPayment($createdPayment);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

    public function backPayment(Request $request)
    {
        try {

            $validator = Validator::make($request->all(), [
                'payment_id' => ['required', 'integer', 'exists:payments,id'],
                'Status' => ['required', 'string']
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ]);
            }

            $status = $validator->validated()['Status'];
            $paymentId = $validator->validated()['payment_id'];

            $payment = Payments::findOrFail($paymentId);
            $amount = $payment->amount;


            DB::beginTransaction();

            if ($status === 'OK') {

                $data = [
                    'merchant_id' => '1ac73777-76ce-45bd-8408-1fa1436f729e',
                    'amount' => $amount,
                    'currency' => 'IRT',
                    'authority' => $request['Authority'],
                ];

                $ch = curl_init();
                curl_setopt($ch, CURLOPT_URL, 'https://api.zarinpal.com/pg/v4/payment/verify.json');
                curl_setopt($ch, CURLOPT_POST, 1);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

                $res = curl_exec($ch);
                curl_close($ch);
                $res = json_decode($res, true, 512, JSON_THROW_ON_ERROR);

                if ($res['data']['code'] == "100" || $res['data']['code'] == "101") {

                    if($payment->action === PaymentActions::TravelPayment){
                        MyService::newUserWalletRecord(
                            $payment->uid,
                            $payment->user->wallet - $payment->amount,
                            'پرداخت آنلاین سفر با کد ' . $payment->travel->travel_code
                        );
                        $payment->driver->increment('wallet', $payment->travel->travel_price);
                        $payment->travel->update(['payment_method' => 'online']);
                    }

                    if ($payment->action === PaymentActions::DriverWallet) {
                        MyService::newDriverWalletRecord(
                            $payment->driver_id,
                            $payment->driver->wallet + $payment->amount,
                            'افزایش کیف پول به صورت آنلاین'
                        );
                        $payment->driver->increment('wallet', $payment->amount);
                    }

                    if ($payment->action === PaymentActions::UserWallet) {
                        MyService::newUserWalletRecord(
                            $payment->uid,
                            $payment->user->wallet + $payment->amount,
                            'افزایش کیف پول به صورت آنلاین'
                        );
                        $payment->user->increment('wallet', $payment->amount);
                    }

                    $payment->update([
                        'status' => PaymentsStatus::Succeeded,
                    ]);

                    DB::commit();
                    return view('zarinpalOK', ['device' => $payment->device]);
                }

                DB::rollBack();
                $payment->update([
                    'answer' => 'code:' .' ' . $res['data']['code'] .' '. 'message:' .' ' . $res['data']['message'],
                    'status' => PaymentsStatus::Failed,
                ]);

                return view('zarinpalNOK', ['device' => $payment->device]);
            }

            DB::rollBack();
            $payment->update([
                'status' => PaymentsStatus::Canceled,
            ]);

            return view('zarinpalNOK', ['device' => $payment->device]);

        }catch (\Exception $e) {
            DB::rollBack();
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }
}
