<?php

namespace App\Http\Controllers\Panel;

use App\Enums\TravelStage;
use App\Http\Controllers\Controller;
use App\Models\Travels;
use App\Services\MyService;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TravelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): object
    {
        try {

            $travels = Travels::all()
                ->map(function ($travel) {

                    $date = MyService::G2J($travel->created_at);

                    return [
                        'id' => $travel->id,
                        'driver_name' => $travel->driver ? $travel->driver->first_name . ' ' . $travel->driver->last_name : 'مشخص نشده',
                        'user_name' => $travel->user->username,
                        'travel_price' => $travel->travel_price,
                        'app_commission' => $travel->app_commission,
                        'travel_date' => $date,
                        'travel_code' => $travel->travel_code,
                        'current_stage' => MyService::travelStringStatus($travel)
                    ];
                });

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $travels
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): object
    {
        try {

            $validator = Validator::make(['id' => $id], [
                'id' => ['required', 'integer', 'exists:travels,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => $validator->errors()->first(),
                ], 422);
            }

            $travel = Travels::findOrFail($id);


            $detailsTravel = $travel->details ?? (object)[];
            if ($travel->details) {
                $detailsTravel->payment_recipient = (bool)$detailsTravel->payment_recipient;
                $detailsTravel->title_load_type = $detailsTravel->load_type_id ? $detailsTravel?->loadType?->title : '';
            }
            $detailsTravel->delay_time = $travel->delay_time;
            $detailsTravel->commuting = (bool)$travel->commuting;
            $detailsTravel->two_destination = ($travel->commuting && count($travel->destinations) > 2) || (!$travel->commuting && count($travel->destinations) > 1);

            $res = (object)[
                'id' => $travel->id,
                'travel_code' => $travel->travel_code,
                'travel_price' => $travel->travel_price,
                'app_commission' => $travel->app_commission,
                'travel_date' => MyService::G2J($travel->created_at),
                'cancel_type' => $travel->cancel_type,
                'cancel_description' => $travel->cancel_description,
                'driver_category' => $travel->driverCategory->name,
                'driver_category_id' => $travel->driver_category,
                'discount_code' => $travel->discount_code,
                'origin_address' => $travel->origin_address,
                'current_stage' => MyService::travelStringStatus($travel),
                'driver' => $travel->driver ? $travel->driver->only(['id', 'first_name', 'last_name', 'mobile', 'address', 'vehicle_color', 'vehicle_name', 'vehicle_plate']) : null,
                'user' => $travel->user ? $travel->user->only(['id', 'username', 'mobile']) : null,
                'destination' => $travel->destinations->map(function ($destination) {
                    return [
                        'order' => $destination->order,
                        'destination_address' => $destination->destination_address,
                    ];
                }),
                'details' => $detailsTravel,
            ];

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $res
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
