<?php

namespace App\Http\Controllers\Panel;

use App\Enums\UserPanelTypes;
use App\Http\Controllers\Controller;
use App\Models\PanelUsers;
use App\Services\MyService;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class PanelUsersController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {

            $users = PanelUsers::all()
            ->map(function ($user) {

                $date = explode(' ' , MyService::G2J($user->created_at));

                return [
                    'id' => $user->id,
                    'username' => $user->username,
                    'mobile' => $user->mobile,
                    'is_blocked' => $user->is_blocked,
                    'created' => $date[0],
                ];
            });


            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $users
            ]);

        }catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {

            $validator = Validator::make($request->all(), [
                'username' => ['required', 'string', 'max:255'],
                'mobile' => ['required', 'string', 'max:255', 'unique:panel_users'],
                'type' => ['required', 'string', Rule::in(array_column(UserPanelTypes::cases(), 'value'))],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ],422);
            }

            $data = $validator->validated();

            $data['code'] = random_int(100000, 999999);
            $data['expires_code'] = now()->addMinutes();

            PanelUsers::create($data);

            return response()->json([
                'success' => true,
                'message' => 'کاربر جدید با موفقیت ایجاد شد'
            ]);

        }catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {

            $user = PanelUsers::find($id);

            unset($user['code'] , $user['expires_code']);

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $user
            ]);

        }catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {

            $validator = Validator::make($request->all(), [
                'username' => ['sometimes', 'string', 'max:255'],
                'is_blocked' => ['sometimes', 'boolean'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ]);
            }

            $data = $validator->validated();

            PanelUsers::find($id)->update($data);

            return response()->json([
                'success' => true,
                'message' => 'کاربر  با موفقیت ویرایش  شد'
            ]);

        }catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): object
    {
        try {

            PanelUsers::find($id)->delete();

            return response()->json([
                'success' => true,
                'message' => 'کاربر  با موفقیت حذف شد'
            ]);

        }catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }
}
