<?php

namespace App\Http\Controllers\Panel;

use App\Http\Controllers\Controller;
use App\Models\DiscountCodes;
use App\Models\UsersDiscountCodes;
use App\Services\MyService;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class DiscountCodeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): object
    {
        try {

            $discounts = DiscountCodes::all()
                ->map(function ($item) {

                    $expiredDate = explode(" ", MyService::G2J($item->expired_at));

                    return [
                        'id' => $item->id,
                        'code' => $item->code,
                        'discount' => $item->discount,
                        'expired_at' => $expiredDate[0],
                        'minimum_usage_price' => $item->minimum_usage_price,
                        'count' => $item->count,
                        'is_active' => $item->is_active,
                        'expired' => $item->is_expired ? 'منقضی شده' : 'فعال',
                    ];
                });

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $discounts
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'status_use' => ['required', 'string', 'in:public,users'],
                'status_code' => ['required', 'string', 'in:percentage,price'],
                'code' => ['required', 'string', 'max:7', 'unique:discount_codes,code'],
                'discount_price' => ['required_if:status_code,price', 'exclude_if:status_code,percentage', 'numeric', 'min:1000', 'max:1000000'],
                'discount_percentage' => ['required_if:status_code,percentage', 'exclude_if:status_code,price', 'numeric', 'between:1,100'],
                'minimum_usage_price' => ['required', 'numeric', 'min:1000', 'max:1000000'],
                'count' => ['required', 'numeric', 'min:1', 'max:10000'],
                'expired_at' => ['required', 'date', 'after:today', 'date_format:Y-m-d'],
                'select_users' => ['required_if:status_use,users', 'array'],
                'select_users.*' => ['nullable', 'integer', 'exists:users,id'],
            ]);


            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $statusUse = $request['status_use'];
            $selectUsers = $request['select_users'];

            DB::beginTransaction();

            $data = $request->only(['code', 'discount_price', 'discount_percentage', 'minimum_usage_price', 'count', 'expired_at']);
            $data['is_public'] = $statusUse === 'public';
            $data['is_selected_users'] = $statusUse === 'users';

            $createDiscount = DiscountCodes::create($data);

            if (!$createDiscount) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'متاسفانه کد تخفیف ساخته نشد !'
                ]);
            }

            if ($statusUse === 'users') {
                foreach ($selectUsers as $item) {
                    $data = [
                        'uid' => $item,
                        'code_id' => $createDiscount->id,
                    ];

                    if (!UsersDiscountCodes::create($data)) {
                        DB::rollBack();
                        return response()->json([
                            'success' => false,
                            'message' => 'متاسفانه مشکلی پیش آمده است !'
                        ]);
                    }
                }
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'کد تخفیف شما با موفقیت ایجاد شد'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): object
    {
        try {

            $discount = DiscountCodes::findOrFail($id);


           $res = (object)[
               'id' => $discount->id,
               'code' => $discount->code,
               'discount_price' => $discount->discount_price,
               'discount_percentage' => $discount->discount_percentage,
               'minimum_usage_price' => $discount->minimum_usage_price,
               'count' => $discount->count,
               'expired_at' => $discount->expired_at,
               'is_active' => $discount->is_active,
               'select_users' => $discount->selectUsers->map(function ($item) {return $item->uid;}),
               'status_use' => $discount->is_public ? 'public' : 'users',
               'status_code' => $discount->discount_percentage ? 'percentage' : 'price',
           ];

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $res
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'status_use' => ['sometimes', 'required', 'string', 'in:public,users'],
                'status_code' => ['sometimes', 'required', 'string', 'in:percentage,price'],
                'is_active' => ['sometimes', 'boolean'],
                'code' => ['sometimes', 'required', 'string', 'max:7', Rule::unique('discount_codes')->ignore($id)],
                'discount_price' => ['sometimes', 'required_if:status_code,price', 'exclude_if:status_code,percentage', 'numeric', 'min:1000', 'max:1000000'],
                'discount_percentage' => ['sometimes', 'required_if:status_code,percentage', 'exclude_if:status_code,price', 'numeric', 'between:1,100'],
                'minimum_usage_price' => ['sometimes', 'required', 'numeric', 'min:1000', 'max:1000000'],
                'count' => ['sometimes', 'required', 'numeric', 'min:1', 'max:10000'],
                'expired_at' => ['sometimes', 'required', 'date', 'after:today', 'date_format:Y-m-d'],
                'select_users' => ['required_if:status_use,users', 'array'],
                'select_users.*' => ['nullable', 'integer', 'exists:users,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $discount = DiscountCodes::findOrFail($id);

            $updateData = $request->only([
                'code',
                'discount_price',
                'discount_percentage',
                'minimum_usage_price',
                'count',
                'is_active',
                'expired_at',
            ]);

            $updateData = array_filter($updateData, function ($value) {
                return !is_null($value);
            });

            $discount->update($updateData);

            $selectUsers = $validator->validated()['select_users'] ?? [];

            if (is_array($selectUsers)) {

                $existingUids = $discount->selectUsers()->pluck('uid')->toArray();

                $toInsert = array_diff($selectUsers, $existingUids);

                $toDelete = array_diff($existingUids, $selectUsers);

                if (!empty($toInsert)) {
                    $insertData = array_map(fn($uid) => [
                        'code_id' => $discount->id,
                        'uid' => $uid
                    ], $toInsert);

                    UsersDiscountCodes::insert($insertData);
                }

                if (!empty($toDelete)) {
                    UsersDiscountCodes::where('code_id', $discount->id)
                        ->whereIn('uid', $toDelete)
                        ->delete();
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد'
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
