<?php

namespace App\Http\Controllers\Driver;

use App\Enums\TravelDestinationStatus;
use App\Enums\TravelStage;
use App\Events\AcceptedRequest;
use App\Events\RemoveRequest;
use App\Http\Controllers\Controller;
use App\Models\AppConfig;
use App\Models\DriverWalletHistory;
use App\Models\Travels;
use App\Services\MyService;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;


class TravelController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): object
    {
        try {
            $driver = auth('driver')->user();

            try {
                MyService::checkDebtorDriver($driver);
            } catch (\Exception $e) {
                return $e->response();
            }

            $radius = Cache::rememberForever('passenger_receiving_radius', function () {
                return AppConfig::where('title', 'passenger_receiving_radius')->value('content');
            });

            $currentLat = $driver->current_lat;
            $currentLng = $driver->current_lng;
            $driverCategory = $driver->category_id;

            $raw = "(6371 * acos(cos(radians($currentLat)) * cos(radians(origin_lat)) * cos(radians(origin_lng) - radians($currentLng)) + sin(radians($currentLat)) * sin(radians(origin_lat))))";

            $travels = Travels::query()
                ->latest()
                ->where('is_running', false)
                ->where('is_canceled', false)
                ->where('is_completed', false)
                ->where('is_unsuccessful', false)
                ->where('driver_category', $driverCategory)
                ->whereRaw("$raw < ?", [$radius + 2]) // کمی بزرگتر از radius چون این مسافت مستقیمه
                ->get()
                ->map(function ($travel) use ($driver, $radius) {

                    $currentTravel = MyService::estimatedDrivingTime(
                        $driver->current_lat,
                        $driver->current_lng,
                        $travel->origin_lat,
                        $travel->origin_lng,
                    );

                    if ($currentTravel && isset($currentTravel['estimated_km']) && $currentTravel['estimated_km'] < $radius) {

                        $detailsTravel = $travel->details ?? (object)[];
                        if ($travel->details) {
                            $detailsTravel->load_type_title = $detailsTravel->load_type_id ? $detailsTravel?->loadType?->title : '';
                            if ($detailsTravel->loadType) {
                                unset($detailsTravel->loadType);
                            }
                        }
                        $detailsTravel->delay_time = $travel->delay_time;
                        $detailsTravel->commuting = (bool)$travel->commuting;
                        $detailsTravel->two_destination = ($travel->commuting && count($travel->destinations) > 2) || (!$travel->commuting && count($travel->destinations) > 1);

                        if ($currentTravel && isset($currentTravel['estimated_km']) && $currentTravel['estimated_km'] < $radius) {

                            $detailsTravel = $travel->details ?? (object)[];

                            $detailsTravel->delay_time = $travel->delay_time;
                            $detailsTravel->commuting = (bool)$travel->commuting;
                            $detailsTravel->two_destination = ($travel->commuting && count($travel->destinations) > 2) || (!$travel->commuting && count($travel->destinations) > 1);


                            $createdTime = explode(" ", MyService::G2J($travel->created_at));

                            return [
                                'create_time' => $createdTime[1],
                                'user_id' => $travel->uid,
                                'travel_id' => $travel->id,
                                'travel_code' => $travel->travel_code,
                                'origin_time' => $currentTravel['estimated_time'],
                                'origin_address' => $travel->origin_address,
                                'type' => $travel->driverCategory->type,
                                'vehicle' => $travel->driverCategory->vehicle,
                                'routes' => $travel->destinations->map(function ($item) {
                                    return [
                                        'distance_km' => $item->distance_km,
                                        'address' => $item->destination_address
                                    ];
                                }),
                                'total' => [
                                    'distance_km' => $travel->destinations->sum('distance_km'),
                                    'price' => $travel->travel_price + $travel->app_commission,
                                ],
                                'details_travel' => $detailsTravel,
                                'user' => (object)[
                                    'username' => $travel->user->username,
                                    'mobile' => $travel->user->mobile,
                                ],
                            ];

                        }

                    }


                });


            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $travels
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $travel = Travels::find($id);


        return MyService::travelStatus($travel);
    }

    /**
     * Update the specified resource in storage.
     */

    public function update(Request $request, string $id)
    {
        try {
            $validator = Validator::make(['id' => $id], [
                'id' => ['required', 'integer', 'exists:travels,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $travel = Travels::with('destinations')->find($id);
            $buttons = MyService::travelStatus($travel);
            $driver = auth('driver')->user();

            if (!is_null($travel->driver_id) && $travel->driver_id !== $driver->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'احراز هویت شما تایید نشد !'
                ], 401);
            }

            if ($buttons['action_button'] !== $request->input('action_button')) {
                return response()->json([
                    'success' => false,
                    'message' => 'درخواست مورد نظر قابل اجرا نمی باشد!'
                ], 403);
            }

            DB::beginTransaction();

            $okSwitch = true;

            switch ($request->input('action_button')) {
                case 'accepted':

                    try {
                        MyService::checkDebtorDriver($driver);
                    } catch (\Exception $e) {
                        return $e->response();
                    }

                    $updateTravel = $travel->update([
                        'current_stage' => TravelStage::BeforeOrigin,
                        'driver_id' => $driver->id,
                        'is_running' => true
                    ]);

                    $updateDriver = $driver->update([
                        'is_available' => false
                    ]);

                    if (!$updateTravel || !$updateDriver) {
                        $okSwitch = false;
                    }

                    $drivers = MyService::getDriversNearby($travel->driver_category, $travel->origin_lat, $travel->origin_lng);

                    foreach ($drivers as $driver) {
                        MyService::socketRemoveRequest($travel, $driver->id);
                    }

                    MyService::socketAcceptedRequest($travel, $driver->id);

                    break;

                case'arrive_origin':
                    $updateTravel = $travel->update(['current_stage' => TravelStage::AtOrigin]);

                    if (!$updateTravel) {
                        $okSwitch = false;
                    }

                    break;

                case 'start_travel':
                    $updateDest = $travel->destinations->where('order', 1)->first()->update([
                        'status' => TravelDestinationStatus::Running,
                    ]);

                    $updateTravel = $travel->update([
                        'current_stage' => TravelStage::ToDestination,
                        'payment_method' => 'cash',
                    ]);

                    if (!$updateTravel || !$updateDest) {
                        $okSwitch = false;
                    }

                    break;

                case 'arrive_destination':
                    $current = $travel->destinations->where('status', TravelDestinationStatus::Running)->first();
                    $updateCurrent = $current->update(['status' => TravelDestinationStatus::Arrived]);
                    $updateTravel = $travel->update(['current_stage' => TravelStage::AtDestination]);

                    if (!$updateTravel || !$updateCurrent) {
                        $okSwitch = false;
                    }

                    break;

                case 'next_destination':
                    $current = $travel->destinations->where('status', TravelDestinationStatus::Arrived)->first();
                    $next = $travel->destinations->where('order', $current->order + 1)->first();

                    $updateCurrent = $current->update(['status' => TravelDestinationStatus::Completed]);
                    $updateNext = $next->update(['status' => TravelDestinationStatus::Running]);
                    $updateTravel = $travel->update(['current_stage' => TravelStage::ToNextDestination]);

                    if (!$updateTravel || !$updateNext || !$updateCurrent) {
                        $okSwitch = false;
                    }

                    break;

                case 'start_return':
                    $arrived = $travel->destinations->where('status', TravelDestinationStatus::Arrived)->first();
                    $last = $travel->destinations->last();


                    $updateArrived = $arrived->update(['status' => TravelDestinationStatus::Completed]);
                    $updateLast = $last->update(['status' => TravelDestinationStatus::Running]);
                    $updateTravel = $travel->update(['current_stage' => TravelStage::ToOrigin]);

                    if (!$updateTravel || !$updateArrived || !$updateLast) {
                        $okSwitch = false;
                    }

                    break;

                case 'complete_travel':

                    $updateTravel = $travel->update([
                        'current_stage' => TravelStage::Completed,
                        'is_completed' => true,
                        'is_running' => false
                    ]);

                    $updateLastDestination = $travel->destinations()->update([
                        'status' => TravelDestinationStatus::Completed
                    ]);


                    $dateWalletHistory = [
                        'driver_id' => $driver->id,
                        'action' => $travel->payment_method === 'online' ? 'input' : 'output',
                        'amount' => $travel->payment_method === 'online' ? $travel->travel_price : $travel->app_commission,
                        'travel_id' => $travel->id,
                        'description' => $travel->payment_method === 'online' ? 'پرداخت از کیف پول توسط کاربر' : 'پرداخت نقدی توسط کاربر',
                        'wallet_inventory' => $driver->wallet,
                    ];

                    $createdWalletHistory = DriverWalletHistory::create($dateWalletHistory);

                    $updateDriver = $driver->update([
                        'is_available' => true,
                        'wallet' => $travel->payment_method === 'online' ? $driver->wallet + $travel->travel_price : $driver->wallet - $travel->app_commission
                    ]);


                    if (!$updateTravel || !$updateDriver || !$updateLastDestination || !$createdWalletHistory) {
                        $okSwitch = false;
                    }

                    break;
            }

            if (!$okSwitch) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'متاسفانه مشکلی پیش آمده است!'
                ], 500);
            }

            MyService::socketChangeRequestForUser($travel);

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => MyService::travelStatus($travel)
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            app()[ExceptionHandler::class]->report($e);
            return response()->json(['success' => false,
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید!',
                'error' => config('app.debug') ? $e->getMessage() : null], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
