<?php

namespace App\Http\Controllers\Driver;

use App\Enums\TravelStage;
use App\Events\CancelRequestFortDriver;
use App\Events\CancelRequestForUser;
use App\Http\Controllers\Controller;
use App\Models\DriverBankCards;
use App\Models\DriverUser;
use App\Models\DriverWalletWithdrawals;
use App\Models\Travels;
use App\Services\MyService;
use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Morilog\Jalali\Jalalian;

class MiscController extends Controller
{

    public function travelRunning(): object
    {
        try {

            $driver = auth('driver')->user();

            $travel = Travels::where('driver_id', $driver->id)
                ->where('is_running', 1)
                ->latest()
                ->first();

            $res = (object)[];

            if ($travel) {

                $currentTravel = MyService::estimatedDrivingTime(
                    $driver->current_lat,
                    $driver->current_lng,
                    $travel->origin_lat,
                    $travel->origin_lng,
                );

                $detailsTravel = $travel->details ?? (object)[];

                $detailsTravel->delay_time = $travel->delay_time;
                $detailsTravel->commuting = (bool)$travel->commuting;
                $detailsTravel->two_destination = ($travel->commuting && count($travel->destinations) > 2) || (!$travel->commuting && count($travel->destinations) > 1);

                $res = [
                    'user_id' => $travel->uid,
                    'travel_id' => $travel->id,
                    'travel_code' => $travel->travel_code,
                    'origin_time' => $currentTravel['estimated_time'],
                    'origin_lat' => $travel->origin_lat,
                    'origin_lng' => $travel->origin_lng,
                    'origin_address' => $travel->origin_address,
                    'routes' => $travel->destinations->map(function ($item) {
                        return [
                            'distance_km' => $item->distance_km,
                            'address' => $item->destination_address,
                            'lat' => $item->lat,
                            'lng' => $item->lng,
                        ];
                    }),

                    'total' => [
                        'distance_km' => $travel->destinations->sum('distance_km'),
                        'price' => $travel->travel_price + $travel->app_commission,
                    ],

                    'payment' => [
                        'method' => $travel->payment_method,
                    ],

                    'details_travel' => $detailsTravel,

                    'user' => (object)[
                        'username' => $travel->user->username,
                        'mobile' => $travel->user->mobile,
                    ],

                    'status' => MyService::travelStatus($travel)
                ];

            }


            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'is_running' => (bool)$travel,
                'response' => $res
            ]);


        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    public function travelCanceled(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'cancel_title' => ['required', 'string', 'max:255'],
                'cancel_description' => ['nullable', 'string', 'max:255'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $driverId = auth('driver')->id();

            $travel = Travels::where('driver_id', $driverId)
                ->where('is_running', true)
                ->latest()->first();


            if (!$travel) {
                return response()->json([
                    'success' => false,
                    'message' => 'درحال حاضر سفر فعالی ندارید !'
                ], 403);
            }

            if ($travel->has_started) {
                return response()->json([
                    'success' => false,
                    'message' => 'سفر شما درحال انجام است و قابلیت لغو سفر امکان پذیر نمی باشد !'
                ], 403);
            }

            $cancelTitle = $request->input('cancel_title');
            $cancelDescription = $request->input('cancel_description');

            if ($cancelDescription) {
                $cancelTitle .= ' ' . '-' . ' ' . $cancelDescription;
            }

            $travel->cancel_description = $cancelTitle;
            $travel->is_canceled = true;
            $travel->is_running = false;
            $travel->cancel_type = 'driver';
            $travel->save();

            $travel->driver->is_available = true;
            $travel->driver->save();

            MyService::socketCancelRequestFortUser($travel, 'راننده سفر خود را به علت ' . $cancelTitle . ' لغو کرد !');

            return response()->json([
                'success' => true,
                'message' => 'با موفقیت لغو شد' . ' ' . $travel->travel_code . ' ' . 'سفر شما با کد'
            ]);


        } catch (Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    public function dailyReports(): object
    {
        try {

            $driver = auth('driver')->user();

            $todayTehran = Carbon::now('Asia/Tehran');

            $todayJalali = Jalalian::fromCarbon($todayTehran);

            $startOfDay = $todayJalali->toCarbon()->startOfDay()->timezone('UTC');
            $endOfDay = $todayJalali->toCarbon()->endOfDay()->timezone('UTC');

            $travels = Travels::where('driver_id', $driver->id);

            $activeTravel = $travels->where('current_stage', '!==', TravelStage::Completed)->latest()->first();

            $travelsProfit = $travels->where('is_completed', 1)
                ->whereBetween('created_at', [$startOfDay, $endOfDay])
                ->get();


            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => [
                    'profile' => $driver->profile,
                    'travel_profit' => $travelsProfit->sum(function ($travel) {
                        return $travel->travel_price - $travel->app_commission;
                    }),
                    'travel_count' => $travels->count(),
                    'is_online' => $driver->is_online,
                    'active_travel' => $activeTravel ? $activeTravel->id : null,
                    'debtor' => $driver->wallet < 0
                ]
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    public function travelReports(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'one_date' => ['sometimes', 'date_format:Y-m-d'],
                'start_date' => ['sometimes', 'date_format:Y-m-d'],
                'end_date' => ['sometimes', 'nullable', 'date_format:Y-m-d'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }

            if ($request['one_date']) {
                $startDate = Carbon::createFromFormat('Y-m-d', $request['one_date'])->startOfDay();
                $endDate = Carbon::createFromFormat('Y-m-d', $request['one_date'])->endOfDay();
            }

            if ($request['start_date']) {
                $startDate = Carbon::createFromFormat('Y-m-d', $request['start_date'])->startOfDay();
                $endDate = Carbon::createFromFormat('Y-m-d', $request['end_date'] ?? now()->format('Y-m-d'))->endOfDay();
            }

            $travelsDriver = Travels::where('driver_id', auth('driver')->id())
                ->where('is_completed', true)
                ->periodFilter($startDate ?? null, $endDate ?? null)
                ->get();

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => [
                    'count' => $travelsDriver->count(),
                    'amount' => $travelsDriver->sum('travel_price'),
                ]
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    public function driverWalletWithdrawal(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'bank_card_id' => ['nullable', 'integer', 'exists:driver_bank_cards,id'],
                'amount' => ['nullable', 'numeric', 'min:10000'],

                // اگر bank_card_id نبود، این‌ها الزامی‌ان
                'bank_name' => ['required_without:bank_card_id', 'string', 'max:255'],
                'shaba_number' => ['required_without:bank_card_id', 'numeric', 'digits:24'],
                'card_number' => ['required_without:bank_card_id', 'numeric', 'digits:16'],
                'card_name' => ['required_without:bank_card_id', 'string', 'max:255'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $bank_card_id = $request['bank_card_id'];
            $driver = auth('driver')->user();
            $amount = $request['amount'] ?? $driver->wallet;

            if ($amount <= 0 || $amount > $driver->wallet) {
                return response()->json([
                    'success' => false,
                    'message' => 'کیف پول شما موجودی کافی جهت تسویه ندارد !'
                ], 403);
            }

            if ($bank_card_id) {
                $card = DriverBankCards::find($bank_card_id);

                try {
                    MyService::checkRealPerson($card->driver_id, 'driver');
                } catch (\Exception $e) {
                    return $e->response();
                }
            }


            if (!$bank_card_id) {

                $data = $request->only('bank_name', 'card_number', 'card_name', 'shaba_number');

                $bankCardController = app()->make(BankCardsController::class);
                $storeResponse = $bankCardController->store($data);

                $createdCard = $storeResponse->getData();

                if (!$createdCard || !$createdCard->success) {
                    return response()->json([
                        'success' => false,
                        'message' => 'ثبت کارت بانکی با خطا مواجه شد !'
                    ], 500);
                }

                $bank_card_id = $createdCard->id;
            }


            $createWithdrawal = DriverWalletWithdrawals::create([
                'driver_id' => $driver->id,
                'bank_card_id' => $bank_card_id,
                'amount' => $amount,
            ]);

            if ($createWithdrawal) {
                return response()->json([
                    'success' => true,
                    'message' => 'عملیات با موفقیت انجام شد'
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'عملیات انجام نشد !'
            ], 500);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * @throws ValidationException
     */
    public function setLocation(Request $request): object
    {

        try {

            $validator = Validator::make($request->all(), [
                'current_lat' => ['required', 'numeric', 'between:-90,90'],
                'current_lng' => ['required', 'numeric', 'between:-180,180'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            DriverUser::where('id', auth('driver')->id())->update($validator->validated());

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد'
            ]);


        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

}
