<?php

namespace App\Http\Controllers\Driver;

use App\Enums\TravelStage;
use App\Http\Controllers\Controller;
use App\Models\DriverUser;
use App\Models\Travels;
use App\Rules\CheckGuardOperation;
use App\Services\MyService;
use Carbon\Carbon;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Morilog\Jalali\Jalalian;

class DriverController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): object
    {
        try {

            $drivers = DriverUser::all()
                ->map(function ($driver) {

                    $date = explode(' ' , MyService::G2J($driver->created_at));

                    return [
                        'id' => $driver->id,
                        'first_name' => $driver->first_name,
                        'last_name' => $driver->last_name,
                        'mobile' => $driver->mobile,
                        'is_blocked' => $driver->is_blocked,
                        'is_confirmation' => $driver->is_confirmation,
                        'created' => $date[0],
                        'state' => $driver->state?->name,
                        'city' => $driver->city?->name,
                        'wallet' => $driver->wallet,
                    ];
                });


            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $drivers
            ]);

        }catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): object
    {

        try {
            $validator = Validator::make($request->all(), [
                'first_name' => ['required', 'string', 'min:3', 'max:255'],
                'last_name' => ['required', 'string', 'min:3', 'max:255'],
                'mobile' => ['required', 'regex:/^09\d{9}$/', 'unique:driver_users,mobile'],
                'national_code' => ['required', 'numeric', 'digits:10', 'unique:driver_users,national_code'],
                'birth_certificate_number' => ['required', 'numeric'],
                'father_name' => ['required', 'string', 'max:255'],
                'birth_date' => ['required', 'date'],
                'state_id' => ['required', 'integer', Rule::exists('states', 'id')->whereNull('parent_id'),],
                'city_id' => ['required', Rule::exists('states', 'id')->where('parent_id', $request->state_id),],
                'address' => ['required', 'string', 'max:255'],
                'postal_code' => ['required', 'numeric', 'digits:10'],
                'landline_phone' => ['required', 'numeric', 'digits:11'],
                'category_id' => ['required', 'integer', Rule::exists('driver_categories', 'id')->where('is_active', 1),],
                'profile' => ['required', 'string', 'max:255'],
                'vehicle_name' => ['required', 'string', 'max:255'],
                'vehicle_color' => ['required', 'string', 'max:255'],
                'vehicle_plate' => ['required', 'array'],
                'vin_number' => ['required', 'string', 'digits:17'],
                'is_vehicle_owner' => ['required', 'boolean'],
                'vehicle_card_front_image' => ['required', 'string', 'max:255'],
                'vehicle_card_back_image' => ['required', 'string', 'max:255'],
                'driving_license_front_image' => ['required', 'string', 'max:255'],
                'driving_license_back_image' => ['required', 'string', 'max:255'],
                'insurance_code' => ['required', 'numeric', 'digits:11'],
                'insurance_national_code' => ['required', 'numeric', 'digits:10'],
                'insurance_expiry_date' => ['required', 'date'],
                'face_verification_image' => ['required', 'string', 'max:255'],
                'signature_verification_image' => ['required', 'string', 'max:255'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }


            $data = $validator->validated();

            $data['vehicle_plate'] = json_encode($data['vehicle_plate'], JSON_THROW_ON_ERROR);

            $driver = DriverUser::create($data);

            $token = $driver->createToken('auth_token', ['driver'])->plainTextToken;

            return response()->json([
                'success' => true,
                'message' => 'ثبت ‌نام با موفقیت انجام شد',
                'token' => $token,
                'uid' => $driver->id
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): object
    {
        try {

            $validator = Validator::make(['id' => $id], [
                'id' => ['required', 'integer', 'exists:driver_users,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            try {
                MyService::checkRealPerson($id, 'driver');
            } catch (\Exception $e) {
                return $e->response();
            }

            $driver = DriverUser::find($id);

            unset($driver['fcm_token']);
            $driver->is_vehicle_owner = (bool)$driver->is_vehicle_owner;

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $driver
            ]);


        }catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): object
    {
        try {
            $validator = Validator::make($request->all(), [
                'first_name' => ['sometimes', 'required', 'string', 'min:3', 'max:255'],
                'last_name' => ['sometimes', 'required', 'string', 'min:3', 'max:255'],
                'profile' => ['sometimes', 'required', 'string', 'min:3', 'max:255'],
                'is_online' => ['sometimes', 'required', 'boolean'],
                'father_name' => ['sometimes', 'required', 'string', 'min:3', 'max:255', new CheckGuardOperation],
                'is_confirmation' => ['sometimes', 'required', 'in:0,1,-1', new CheckGuardOperation],
                'is_blocked' => ['sometimes', 'required', 'boolean', new CheckGuardOperation],
                'wallet' => ['sometimes', 'numeric', new CheckGuardOperation],
                'current_lat' => ['required_with:current_lng', 'nullable', 'numeric', 'between:-90,90'],
                'current_lng' => ['required_with:current_lat', 'nullable', 'numeric', 'between:-180,180'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            try {
                MyService::checkRealPerson($id, 'driver');
            } catch (\Exception $e) {
                return $e->response();
            }

            DB::beginTransaction();

            $driver = DriverUser::findOrFail($id);
            $data = $validator->validated();

            if (isset($data['profile'])) {
                $newProfile = basename($data['profile']);

                if ($newProfile !== basename($driver->profile)) {
                    // حذف تصویر قبلی از فایل‌سیستم
                    $oldPath = storage_path('app/public/uploads/' . basename($driver->profile));
                    if (file_exists($oldPath)) {
                        unlink($oldPath);
                    }

                    $data['profile'] = $newProfile;
                } else {
                    unset($data['profile']);
                }
            }

            if(isset($data['wallet'])){
                $createdWalletHistory = MyService::newDriverWalletRecord($id, $data['wallet'], 'تغییر از پنل مدیریت');

                if (!$createdWalletHistory) {
                    DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'متاسفانه مشکلی پیش آمده است !'
                    ], 500);
                }
            }

            if (!$driver->update($data)) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'متاسفانه مشکلی پیش آمده است !'
                ], 500);
            }

            DB::commit();
            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
