<?php

namespace App\Http\Controllers\Common;

use App\Http\Controllers\Controller;
use App\Models\LoadTypes;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LoadTypeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): object
    {
        try {

            $isPanel = auth('panel')->check();

            $loadTypes = LoadTypes::where('is_deleted', false)
                ->when(!$isPanel, fn($q) => $q->where('is_active', true))
                ->get()
                ->map(fn($loadType) => $isPanel ? [
                    'id' => $loadType->id,
                    'title' => $loadType->title,
                    'is_active' => $loadType->is_active,
                ] : [
                    'id' => $loadType->id,
                    'value' => $loadType->id,
                    'label' => $loadType->title,
                ]);

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $loadTypes
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'title' => ['required', 'string', 'min:3', 'max:255'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $created = LoadTypes::create($validator->validated());

            if ($created) {
                return response()->json([
                    'success' => true,
                    'message' => 'عملیات با موفقیت انجام شد'
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'عملیات انجام نشد !'
            ], 500);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'title' => ['sometimes', 'string', 'min:3', 'max:255'],
                'is_active' => ['sometimes', 'boolean'],
                'is_deleted' => ['sometimes', 'boolean'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first(),
                ], 422);
            }

            $updated = LoadTypes::where('id', $id)->update($validator->validated());

            if ($updated) {
                return response()->json([
                    'success' => true,
                    'message' => 'عملیات با موفقیت انجام شد'
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'عملیات انجام نشد !'
            ], 500);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): object
    {
        try {

            $validator = Validator::make(['id' => $id], [
                'id' => ['required', 'integer', 'exists:load_types,id'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $type = LoadTypes::where('id', $id)->update(['is_deleted' => true]);

            if ($type) {
                return response()->json([
                    'success' => true,
                    'message' => 'عملیات با موفقیت انجام شد'
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'عملیات انجام نشد !'
            ], 500);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }
}
