<?php

namespace App\Http\Controllers\Common;

use App\Http\Controllers\Controller;
use App\Models\AppConfig;
use App\Models\LoadTypes;
use App\Models\States;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CommonController extends Controller
{
    public function states():object
    {
        try {

            $states = States::whereNull('parent_id')
                ->get()
                ->map(function ($state) {
                    return [
                        'id' => $state->id,
                        'name' => $state->name,
                        'cities' => $state->cities->map(function ($city) {
                            return [
                                'id' => $city->id,
                                'name' => $city->name,
                            ];
                        }),
                    ];
                });

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => $states
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }
    }

    public function legalPages(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'type' => ['required', 'string', 'in:user,driver'],
                'action' => ['required', 'string', 'in:about_us,privacy_policy,rules']
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => $validator->errors()->first(),
                ],422);
            }

            $title = $validator->getData()['type'] . '_' . $validator->getData()['action'];

            $about = AppConfig::where('title', $title)->first();

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => [
                    'content' => $about->content,
                ]
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

    public function version(Request $request): object
    {
        try {

            $validator = Validator::make($request->all(), [
                'type' => ['required', 'string', 'in:user,driver'],
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $validator->errors()->first()
                ], 422);
            }

            $type = $request->input('type') . '_' . 'version';


            $version = AppConfig::where('title', $type)->first();

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => [
                    'content' => $version->content,
                ]
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

    public function supportPhone(): object
    {
        try {

            $phone = AppConfig::where('title', 'support_phone')->first();

            return response()->json([
                'success' => true,
                'message' => 'عملیات با موفقیت انجام شد',
                'response' => [
                    'content' => $phone->content,
                ]
            ]);

        } catch (\Exception $e) {
            app()[ExceptionHandler::class]->report($e);
            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'مشکلی پیش آمده است لطفا مجددا تلاش فرمایید !'
            ], 500);
        }

    }

    public function uploads(Request $request): object
    {

        $validator = Validator::make($request->all(), [
            'file' => ['required', 'file', 'mimes:jpeg,jpg,png', 'max:5000'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $file = $request->file('file');

        $name = time() * random_int(100, 1000) . '_' . $file->getClientOriginalName();
        $name = str_replace(' ', '', $name);

        // ذخیره فایل با نام جدید
        $file->storeAs('uploads', $name, 'public');

        // ارسال مسیر فایل ذخیره شده
        return response()->json([
            'success' => true,
            'file_name' => $name
        ]);
    }
}
